/***************************************************************
 *                   Materials Object Library                  *
 *    Class Schmid_tensor : declaration for Schmid's tensor    *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2008,2009,2012,2014 COLLARD Christophe
 * copyright © 2008,2009,2012,2014 Centre National de la Recherche Scientifique
 * copyright © 2008,2009 Arts et Métiers ParisTech
 * copyright © 2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2012,2014 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::Schmid_tensor
    \brief Computes Schmid's tensor

    \htmlonly
    <FONT color="#838383">

    Schmid_tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Computes the symmetric Schmid tensor associated with \f$ p^\text{th} \f$ slip system: \n
    \f$ R_{ij}^p = (n^p_i m^p_j + n^p_j m^p_i) \f$ \n
    where \f$ \mathbf{n}^p \f$ is the unit vector normal to the slip plane and \f$ \mathbf{m}^p \f$ is the unit vector in the slip direction.

    \author copyright \htmlonly &#169; \endhtmlonly 2008, 2009, 2012, 2014 Christophe COLLARD \n
            copyright \htmlonly &#169; 2008, 2009, 2012, 2014 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
            copyright \htmlonly &#169; 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2012, 2014 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2008-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type Schmid_tensor
#endif

#ifndef __Schmid_tensor_hpp
#define __Schmid_tensor_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __symtensors2_hpp
#include "MOL++/symtensors2.hpp"
#endif

using namespace std;

namespace materiol
{


//===========================================================
template <class T> class Schmid_tensor : public symtensor2<T>
//===========================================================
{
  private :
    vector<T> n_, m_;  // normal to slip plane and slip direction

  public :
    Schmid_tensor () : symtensor2<T> () {} // default constructor
    Schmid_tensor (const vector<T>&, const vector<T>&);  // constructor with data given in argument
    Schmid_tensor (ifstream&);  // constructor with data stored in a file
    ~Schmid_tensor () {}        // destructor

    vector<T> n() {return n_;}
    vector<T> m() {return m_;}
    symtensor2<T> R() {return (T) 0.5 * ( (n_ ^ m_) + (m_ ^ n_));}
};


//=====Private methods for Schmid tensor=======================================

//=====Public methods for slip_system==========================================


/*!
  \brief Computes the symmetric Schmid tensor associated with the \f$ p^\text{th} \f$ slip system

  \f$ \displaystyle R_{ij}^p = \frac{1}{2} (n^p_i m^p_j + n^p_j m^p_i) \f$, \n
  where \f$ \mathbf{n}^p \f$ is the unit vector normal to the habit plane and \f$ \mathbf{m}^p \f$ is the unit vector in the slip direction.
  \param n unit vector normal to the habit plane
  \param m unit vector in the slip direction
*/

//-----------------------------------------------------------------------------------------------------------------------
template <class T> Schmid_tensor<T>::Schmid_tensor (const vector<T>& n, const vector<T>& m) : symtensor2<T> (3, 3, false)
//-----------------------------------------------------------------------------------------------------------------------
{
  assert (n.dim()==3 && m.dim()==3);

  T n_norm = n.norm();
  T m_norm = m.norm();

  assert (n_norm > epsilon && m_norm > epsilon);

  n_ = n / n_norm;
  m_ = m / m_norm;

  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      (*this)(i,j)  = 0.5 * (n_[i] * m_[j] + n_[j] * m_[i]);
}


/*!
  \brief Computes the symmetric Schmid tensor associated with the \f$ p^\text{th} \f$ slip system

  \f$ \displaystyle R_{ij}^p = \frac{1}{2} (n^p_i m^p_j + n^p_j m^p_i) \f$, \n
  where \f$ \mathbf{n}^p \f$ is the unit vector normal to the habit plane and \f$ \mathbf{m}^p \f$ is the unit vector in the slip direction.
  \param file file where \f$ \mathbf{n}^p \f$ and \f$ \mathbf{m}^p \f$ are stored
*/

//-----------------------------------------------------------------------------------------------
template <class T> Schmid_tensor<T>::Schmid_tensor (ifstream& file) : symtensor2<T> (3, 3, false)
//-----------------------------------------------------------------------------------------------
{
  string text;
  for (char ch=file.peek(); !isdigit(ch) && ch!='+' && ch!='-' && ch!='.'; ch=file.peek())
    if (ch == ' ') file.get();
    else getline (file, text);

  vector<T> nl(3), ml(3);
  file >> nl >> ml;
  n_ = nl / (T) nl.norm();
  m_ = ml / (T) ml.norm();
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      (*this)(i,j)  = 0.5 * (n_[i] * m_[j] + n_[j] * m_[i]);
}


}


#endif
